#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "texture.h"
#include "object.h"

static inline Color sample_image(const Image &img, float u, float v, Texture::WrapMode wrapping);

Texture::Texture()
	: sampling(SampleMode::nearest), wrapping(WrapMode::repeat)
{
}

Texture::~Texture() {}

void Texture::set_name(const char *name)
{
	this->name = name;
}

const char *Texture::get_name() const
{
	return name.c_str();
}

void Texture::set_sampling_mode(SampleMode mode)
{
	sampling = mode;
}

Texture::SampleMode Texture::get_sampling_mode() const
{
	return sampling;
}

void Texture::set_wrapping_mode(WrapMode mode)
{
	wrapping = mode;
}

Texture::WrapMode Texture::get_wrapping_mode() const
{
	return wrapping;
}


Color Texture::sample(const HitPoint &hit) const
{
	return sample(hit.texcoord.x, hit.texcoord.y, 0.0f);
}

bool Texture2D::load(const char *fname)
{
	name = fname;
	return img.load(fname);
}

Color Texture2D::sample(float u, float v, float w) const
{
	return sample_image(img, u, v, wrapping);
}

TextureCube::TextureCube()
{
	wrapping = WrapMode::clamp;
}

bool TextureCube::load(const char *fname)
{
	// assume it's a cubemap descriptor file (face path per line)
	FILE *fp;
	if(!(fp = fopen(fname, "r"))) {
		fprintf(stderr, "failed to open the cubemap descriptor %s: %s\n", fname, strerror(errno));
		return false;
	}

	name = fname;

	char *prefix = (char*)alloca(strlen(fname) + 1);
	strcpy(prefix, fname);
	char *ptr = strrchr(prefix, '/');
	if(!ptr) {
		ptr = prefix;
	}
	*ptr = 0;

	int xsz = 0, ysz = 0;

	// load the faces
	char buf[512];
	for(int i=0; i<6; i++) {
		if(!fgets(buf, sizeof buf, fp)) {
			fprintf(stderr, "invalid cubemap descriptor file: %s\n", fname);
			return false;
		}
		if(buf[strlen(buf) - 1] == '\n') {
			buf[strlen(buf) - 1] = 0;
		}

		std::string path = std::string(prefix) + "/" + std::string(buf);
		if(!img[i].load(path.c_str())) {
			fprintf(stderr, "failed to load image: %s\n", path.c_str());
			fclose(fp);
			return false;
		}

		if(i == 0) {
			xsz = img[i].xsz;
			ysz = img[i].ysz;
		} else {
			if(img[i].xsz != xsz || img[i].ysz != ysz) {
				fprintf(stderr, "cubemap %s face image %s size (%dx%d) doesn't match the previous faces (%dx%d)\n",
						fname, path.c_str(), img[i].xsz, img[i].ysz, xsz, ysz);
				fclose(fp);
				return false;
			}
		}
	}
	return true;
}

Color TextureCube::sample(float u, float v, float w) const
{
	int face;
	Vector2 uv;

	float abs_u = fabs(u);
	float abs_v = fabs(v);
	float abs_w = fabs(w);

	if(abs_u > abs_v && abs_u > abs_w) {
		if(u >= 0.0) {
			face = 0;
			uv.x = w / abs_u;
			uv.y = v / abs_u;
		} else {
			face = 1;
			uv.x = -w / abs_u;
			uv.y = v / abs_u;
		}
	} else if(abs_v > abs_w) {
		if(v >= 0.0) {
			face = 2;
			uv.x = u / abs_v;
			uv.y = w / abs_v;
		} else {
			face = 3;
			uv.x = u / abs_v;
			uv.y = -w / abs_v;
		}
	} else {
		if(w >= 0.0) {
			face = 5;
			uv.x = -u / abs_w;
			uv.y = v / abs_w;
		} else {
			face = 4;
			uv.x = u / abs_w;
			uv.y = v / abs_w;
		}
	}

	return sample_image(img[face], uv.x * 0.5 + 0.5, uv.y * 0.5 + 0.5, wrapping);
}

Color TextureCube::sample(const HitPoint &hit) const
{
	return sample(hit.normal.x, hit.normal.y, hit.normal.z);
}

Texture *load_texture(const char *fname)
{
	if(access(fname, R_OK) == -1) {
		fprintf(stderr, "failed to load texture %s: %s\n", fname, strerror(errno));
		return 0;
	}

	Texture2D *tex2d = new Texture2D;
	if(tex2d->load(fname)) {
		return tex2d;
	}
	delete tex2d;

	TextureCube *texcube = new TextureCube;
	if(texcube->load(fname)) {
		return texcube;
	}
	delete texcube;

	return 0;
}

#define CLAMP(x, lo, hi)	((x) < (lo) ? (lo) : ((x) > (hi) ? (hi) : (x)))

static inline Color sample_image(const Image &img, float u, float v, Texture::WrapMode wrapping)
{
	int x = (int)round(u * img.xsz);
	int y = (int)round((1.0 - v) * img.ysz);

	if(wrapping == Texture::WrapMode::clamp) {
		x = CLAMP(x, 0, img.xsz - 1);
		y = CLAMP(y, 0, img.ysz - 1);
	} else {
		x %= img.xsz;
		y %= img.ysz;

		if(x < 0)
			x += img.xsz;
		if(y < 0)
			y += img.ysz;
	}

	return img.pixels[y * img.xsz + x];
}
