/*
Simple introductory ray tracer
Copyright (C) 2012  John Tsiombikas <nuclear@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <stdio.h>
#include "sphere.h"

Sphere::Sphere()
{
	radius = 1.0;
}

Sphere::Sphere(const Vector3 &pos, float rad)
{
	radius = rad;
	this->pos = pos;
}

bool Sphere::intersect(const Ray &inray, HitPoint *pt) const
{
	Ray ray = inray.transformed(inv_xform);

	float a = dot_product(ray.dir, ray.dir);
	float b = 2.0 * ray.dir.x * (ray.origin.x - pos.x) +
		2.0 * ray.dir.y * (ray.origin.y - pos.y) +
		2.0 * ray.dir.z * (ray.origin.z - pos.z);
	float c = dot_product(ray.origin, ray.origin) + dot_product(pos, pos) -
		2.0 * dot_product(ray.origin, pos) - radius * radius;

	float discr = b * b - 4.0 * a * c;
	if(discr < 1e-4)
		return false;

	float sqrt_discr = sqrt(discr);
	float t0 = (-b + sqrt_discr) / (2.0 * a);
	float t1 = (-b - sqrt_discr) / (2.0 * a);

	if(t0 < 1e-4)
		t0 = t1;
	if(t1 < 1e-4)
		t1 = t0;

	float t = t0 < t1 ? t0 : t1;
	if(t < 1e-4)
		return false;

	// fill the HitPoint structure
	pt->obj = this;
	pt->dist = t;
	pt->pos = ray.origin + ray.dir * t;
	pt->normal = (pt->pos - pos) / radius;

	pt->pos.transform(xform);
	pt->normal.transform(dir_xform);
	return true;
}
