#include <stdio.h>
#include <math.h>
#include <map>
#include "scene.h"
#include "sphere.h"
#include "plane.h"
#include "box.h"
#include "light.h"
#include "camera.h"
#include "texture.h"

static bool parse_material(char **argv, Material *mtl, std::string *name);
static Object *parse_sphere(char **argv, const std::map<std::string, Material> &materials);
static Object *parse_plane(char **argv, const std::map<std::string, Material> &materials);
static Object *parse_box(char **argv, const std::map<std::string, Material> &materials);
static Light *parse_light(char **argv);
static Camera *parse_camera(char **argv);
static bool parse_env(char **argv, Scene *scn);
static bool parse_xform(char **argv, Scene *scn);
static bool parse_vec(char **argv, Vector3 *vptr);

static char *strip_space(char *s);
static char **split_args(char *args);

bool load_scene_file(Scene *scn, const char *fname)
{
	FILE *fp;
	if(!(fp = fopen(fname, "rb"))) {
		fprintf(stderr, "%s: failed to open scene file: %s: %s\n", __FUNCTION__, fname, strerror(errno));
		return false;
	}

	char buf[512];
	std::map<std::string, Material> materials;

	int lineno = 0;
	while(fgets(buf, sizeof buf, fp)) {
		char *line = strip_space(buf);

		lineno++;

		if(!*line || *line == '\n' || *line == '#') {
			continue;
		}

		if(strstr(line, "material") == line) {
			Material mtl;
			std::string name;

			char *args = strip_space(line + strlen("material"));
			if(!parse_material(split_args(args), &mtl, &name)) {
				goto err;
			}
			materials[name] = mtl;

		} else if(strstr(line, "sphere") == line) {
			char *args = strip_space(line + strlen("sphere"));
			Object *obj = parse_sphere(split_args(args), materials);
			if(!obj) {
				goto err;
			}
			scn->add_object(obj);

		} else if(strstr(line, "plane") == line) {
			char *args = strip_space(line + strlen("plane"));
			Object *obj = parse_plane(split_args(args), materials);
			if(!obj) {
				goto err;
			}
			scn->add_object(obj);

		} else if(strstr(line, "box") == line) {
			char *args = strip_space(line + strlen("box"));
			Object *obj = parse_box(split_args(args), materials);
			if(!obj) {
				goto err;
			}
			scn->add_object(obj);

		} else if(strstr(line, "light") == line) {
			char *args = strip_space(line + strlen("light"));
			Light *lt = parse_light(split_args(args));
			if(!lt) {
				goto err;
			}
			scn->add_light(lt);

		} else if(strstr(line, "camera") == line) {
			char *args = strip_space(line + strlen("camera"));
			Camera *cam = parse_camera(split_args(args));
			if(!cam) {
				goto err;
			}
			scn->set_camera(cam);

		} else if(strstr(line, "environment") == line) {
			char *args = strip_space(line + strlen("environment"));
			if(!parse_env(split_args(args), scn)) {
				goto err;
			}

		} else if(strstr(line, "xform") == line) {
			char *args = strip_space(line + strlen("xform"));
			if(!parse_xform(split_args(args), scn)) {
				goto err;
			}

		} else {
			fprintf(stderr, "%s error %s:%d: unknown command: %s\n", __FUNCTION__, fname, lineno, line);
			goto err;
		}

	}

	fclose(fp);
	return true;

err:
	fclose(fp);
	return false;
}

bool save_scene_file(const Scene *scn, const char *fname)
{
	FILE *fp;
	if(!(fp = fopen(fname, "wb"))) {
		fprintf(stderr, "%s: failed to save scene file: %s: %s\n", __FUNCTION__, fname, strerror(errno));
		return false;
	}

	for(int i=0; i<scn->get_object_count(); i++) {
		// first write the material
		const Object *obj = scn->get_object(i);
		const Material *mat = &obj->material;

		char name[128];
		sprintf(name, "mat%d", i);

		fprintf(fp, "material -name %s", name);
		fprintf(fp, " -diffuse %.3f %.3f %.3f", mat->diffuse.x, mat->diffuse.y, mat->diffuse.z);
		fprintf(fp, " -specular %.3f %.3f %.3f", mat->specular.x, mat->specular.y, mat->specular.z);
		fprintf(fp, " -shininess %.3f", mat->shininess);
		fprintf(fp, " -emission %.3f %.3f %.3f", mat->emission.x, mat->emission.y, mat->emission.z);
		fprintf(fp, " -reflect %.3f", mat->reflectivity);
		fprintf(fp, " -trans %.3f", mat->transparency);
		fprintf(fp, " -ior %.3f", mat->ior);
		if(mat->tex) {
			fprintf(fp, " -texture %s", mat->tex->get_name());
		}
		fputc('\n', fp);

		// then write the object
		const Sphere *sph;
		const Plane *plane;
		const Box *box;
		if((sph = dynamic_cast<const Sphere*>(obj))) {
			fprintf(fp, "sphere -material %s", name);
			fprintf(fp, " -center %.3f %.3f %.3f", sph->pos.x, sph->pos.y, sph->pos.z);
			fprintf(fp, " -radius %.3f\n", sph->radius);
		} else if((plane = dynamic_cast<const Plane*>(obj))) {
			fprintf(fp, "plane -material %s", name);
			fprintf(fp, " -normal %.3f %.3f %.3f", plane->normal.x, plane->normal.y, plane->normal.z);
			fprintf(fp, " -distance %.3f\n", plane->dist);
		} else if((box = dynamic_cast<const Box*>(obj))) {
			fprintf(fp, "box -material %s", name);
			fprintf(fp, " -min %.3f %.3f %.3f", box->min.x, box->min.y, box->min.z);
			fprintf(fp, " -max %.3f %.3f %.3f\n", box->max.x, box->max.y, box->max.z);
		}
	}

	for(int i=0; i<scn->get_light_count(); i++) {
		const Light *lt = scn->get_light(i);

		fprintf(fp, "light -position %.3f %.3f %.3f", lt->pos.x, lt->pos.y, lt->pos.z);
		fprintf(fp, " -color %.3f %.3f %.3f\n", lt->color.x, lt->color.y, lt->color.z);
	}

	const TargetCamera *tcam = dynamic_cast<const TargetCamera*>(scn->get_camera());
	if(tcam) {
		Vector3 pos = tcam->get_position();
		fprintf(fp, "camera -position %.3f %.3f %.3f", pos.x, pos.y, pos.z);
		Vector3 targ = tcam->get_target();
		fprintf(fp, " -target %.3f %.3f %.3f", targ.x, targ.y, targ.z);
		fprintf(fp, " -fov %.3f\n", tcam->get_fov());
	}


	fclose(fp);
	return true;
}

#define ARGERR(n) \
	do { fprintf(stderr, "failed to parse %s argument\n", (n)); return false; } while(0)

static bool parse_material(char **argv, Material *mtl, std::string *name)
{
	char *endp;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-name") == 0) {
			*name = argv[++i];
		} else if(strcmp(argv[i], "-diffuse") == 0) {
			if(!parse_vec(argv + i + 1, &mtl->diffuse)) {
				ARGERR("diffuse");
			}
			argv += 3;

		} else if(strcmp(argv[i], "-specular") == 0) {
			if(!parse_vec(argv + i + 1, &mtl->specular)) {
				ARGERR("specular");
			}
			argv += 3;

		} else if(strcmp(argv[i], "-emission") == 0) {
			if(!parse_vec(argv + i + 1, &mtl->emission)) {
				ARGERR("emission");
			}
			argv += 3;

		} else if(strcmp(argv[i], "-shininess") == 0) {
			mtl->shininess = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				ARGERR("shininess");
			}

		} else if(strcmp(argv[i], "-reflect") == 0) {
			mtl->reflectivity = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				ARGERR("reflect");
			}

		} else if(strcmp(argv[i], "-trans") == 0) {
			mtl->transparency = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				ARGERR("trans");
			}

		} else if(strcmp(argv[i], "-ior") == 0) {
			mtl->ior = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				ARGERR("ior");
			}

		} else if(strcmp(argv[i], "-texture") == 0) {
			if(!(mtl->tex = load_texture(argv[++i]))) {
				return false;
			}

		} else {
			fprintf(stderr, "invalid material option: %s\n", argv[i]);
			return false;
		}
	}
	return true;
}

static Object *parse_sphere(char **argv, const std::map<std::string, Material> &materials)
{
	char *endp;
	Sphere *sph = new Sphere;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-name") == 0) {
			sph->set_name(argv[++i]);

		} else if(strcmp(argv[i], "-center") == 0) {
			if(!parse_vec(argv + i + 1, &sph->pos)) {
				fprintf(stderr, "failed to parse sphere center vector\n");
				return 0;
			}
			argv += 3;

		} else if(strcmp(argv[i], "-radius") == 0) {
			sph->radius = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse sphere radius\n");
				return 0;
			}

		} else if(strcmp(argv[i], "-material") == 0) {
			auto it = materials.find(argv[++i]);
			if(it == materials.end()) {
				fprintf(stderr, "material %s not found\n", argv[i]);
				return 0;
			}

			sph->material = it->second;
		} else {
			fprintf(stderr, "invalid sphere option: %s\n", argv[i]);
			return 0;
		}
	}

	return sph;
}

static Object *parse_plane(char **argv, const std::map<std::string, Material> &materials)
{
	char *endp;
	Plane *plane = new Plane;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-name") == 0) {
			plane->set_name(argv[++i]);

		} else if(strcmp(argv[i], "-normal") == 0) {
			if(!parse_vec(argv + i + 1, &plane->normal)) {
				fprintf(stderr, "failed to parse plane normal\n");
				return 0;
			}
			argv += 3;

		} else if(strcmp(argv[i], "-distance") == 0) {
			plane->dist = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse plane distance\n");
				return 0;
			}

		} else if(strcmp(argv[i], "-material") == 0) {
			auto it = materials.find(argv[++i]);
			if(it == materials.end()) {
				fprintf(stderr, "material %s not found\n", argv[i]);
				return 0;
			}

			plane->material = it->second;
		} else {
			fprintf(stderr, "invalid plane option: %s\n", argv[i]);
			return 0;
		}
	}

	return plane;
}

static Object *parse_box(char **argv, const std::map<std::string, Material> &materials)
{
	Box *box = new Box;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-name") == 0) {
			box->set_name(argv[++i]);

		} else if(strcmp(argv[i], "-min") == 0) {
			if(!parse_vec(argv + i + 1, &box->min)) {
				fprintf(stderr, "failed to parse box min\n");
				return 0;
			}
			argv += 3;

		} else if(strcmp(argv[i], "-max") == 0) {
			if(!parse_vec(argv + i + 1, &box->max)) {
				fprintf(stderr, "failed to parse box max\n");
				return 0;
			}
			argv += 3;

		} else if(strcmp(argv[i], "-material") == 0) {
			auto it = materials.find(argv[++i]);
			if(it == materials.end()) {
				fprintf(stderr, "material %s not found\n", argv[i]);
				return 0;
			}

			box->material = it->second;
		} else {
			fprintf(stderr, "invalid box option: %s\n", argv[i]);
			return 0;
		}
	}

	return box;
}

static Light *parse_light(char **argv)
{
	Light *lt = new Light;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-position") == 0) {
			if(!parse_vec(argv + i + 1, &lt->pos)) {
				fprintf(stderr, "failed to parse light position\n");
				return 0;
			}
			argv += 3;

		} else if(strcmp(argv[i], "-color") == 0) {
			if(!parse_vec(argv + i + 1, &lt->color)) {
				fprintf(stderr, "failed to parse light color\n");
				return 0;
			}
			argv += 3;

		} else {
			fprintf(stderr, "invalid light option: %s\n", argv[i]);
			return 0;
		}
	}

	return lt;
}

static Camera *parse_camera(char **argv)
{
	char *endp;
	TargetCamera *cam = new TargetCamera;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-position") == 0) {
			Vector3 pos;
			if(!parse_vec(argv + i + 1, &pos)) {
				fprintf(stderr, "failed to parse camera position\n");
				return 0;
			}
			argv += 3;
			cam->set_position(pos);

		} else if(strcmp(argv[i], "-target") == 0) {
			Vector3 targ;
			if(!parse_vec(argv + i + 1, &targ)) {
				fprintf(stderr, "failed to parse camera target\n");
				return 0;
			}
			argv += 3;
			cam->set_target(targ);

		} else if(strcmp(argv[i], "-fov") == 0) {
			float fov = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse camera fov\n");
				return 0;
			}
			cam->set_fov(M_PI * fov / 180.0);

		} else {
			fprintf(stderr, "invalid camera option: %s\n", argv[i]);
			return 0;
		}
	}

	return cam;
}

static bool parse_env(char **argv, Scene *scn)
{
	char *endp;

	TextureCube *env_tex = 0;
	TextureCube *conv_tex = 0;

	float fog_start = -1;
	float fog_end = -1;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-color") == 0) {
			Color bgcolor;
			if(!parse_vec(argv + i + 1, &bgcolor)) {
				fprintf(stderr, "failed to parse environment color\n");
				return false;
			}
			i += 3;
			scn->set_background_color(bgcolor);

		} else if(strcmp(argv[i], "-fog-start") == 0) {
			fog_start = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse environment fog start\n");
				return false;
			}

		} else if(strcmp(argv[i], "-fog-end") == 0) {
			fog_end = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse environment fog end\n");
				return false;
			}

		} else if(strcmp(argv[i], "-texture") == 0 || strcmp(argv[i], "-texture-conv") == 0) {
			Texture *tex = load_texture(argv[++i]);
			if(!tex || !dynamic_cast<TextureCube*>(tex)) {
				fprintf(stderr, "failed to load environment cubemap: %s\n", argv[i]);
				delete tex;
				return false;
			}

			if(strstr(argv[i - 1], "-conv")) {
				conv_tex = (TextureCube*)tex;
			} else {
				env_tex = (TextureCube*)tex;
			}

		} else {
			fprintf(stderr, "invalid environment option: %s\n", argv[i]);
			return false;
		}
	}

	if(env_tex || conv_tex) {
		scn->set_environment_map(env_tex, conv_tex);
	}

	if(fog_start > 0.0 && fog_end > 0.0) {
		scn->set_fog(fog_start, fog_end);
	}

	return true;
}

static bool parse_xform(char **argv, Scene *scn)
{
	char *endp, *name = 0;
	Vector3 pos, rot_axis, scale;
	float rot_angle = 0;
	long tm = 0;
	Extrap extrap = EXTRAP_REPEAT;

	bool have_pos = false;
	bool have_rot_axis = false;
	bool have_rot_angle = false;
	bool have_scale = false;
	bool have_extrap = false;

	for(int i=0; argv[i]; i++) {
		if(strcmp(argv[i], "-name") == 0) {
			name = argv[++i];

		} else if(strcmp(argv[i], "-pos") == 0) {
			if(!parse_vec(argv + i + 1, &pos)) {
				fprintf(stderr, "failed to parse xform position\n");
				return false;
			}
			have_pos = true;
			i += 3;

		} else if(strcmp(argv[i], "-rot-axis") == 0) {
			if(!parse_vec(argv + i + 1, &rot_axis)) {
				fprintf(stderr, "failed to parse xform rotation axis\n");
				return false;
			}
			have_rot_axis = true;
			i += 3;

		} else if(strcmp(argv[i], "-rot-angle") == 0) {
			rot_angle = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse xform rotation angle\n");
				return false;
			}
			rot_angle = M_PI * rot_angle / 180.0;
			have_rot_angle = true;

		} else if(strcmp(argv[i], "-scale") == 0) {
			if(!parse_vec(argv + i + 1, &scale)) {
				fprintf(stderr, "failed to parse xform scale\n");
				return false;
			}
			have_scale = true;
			i += 3;

		} else if(strcmp(argv[i], "-time") == 0) {
			float tm_sec = strtod(argv[++i], &endp);
			if(endp == argv[i]) {
				fprintf(stderr, "failed to parse xform time\n");
				return false;
			}
			tm = (long)(tm_sec * 1000.0f);

		} else if(strcmp(argv[i], "-extrapolation") == 0) {
			if(strcmp(argv[++i], "extend") == 0) {
				extrap = EXTRAP_EXTEND;
			} else if(strcmp(argv[i], "clamp") == 0) {
				extrap = EXTRAP_CLAMP;
			} else if(strcmp(argv[i], "repeat") == 0) {
				extrap = EXTRAP_REPEAT;
			} else if(strcmp(argv[i], "pingpong") == 0) {
				extrap = EXTRAP_PINGPONG;
			} else {
				fprintf(stderr, "failed to parse xform extrapolation, invalid mode: %s\n", argv[i]);
				return false;
			}
			have_extrap = true;

		} else {
			fprintf(stderr, "invalid xform option: %s\n", argv[i]);
			return false;
		}
	}

	if(!name) {
		fprintf(stderr, "invalid xform command, missing -name option\n");
		return false;
	}
	if(have_rot_angle != have_rot_axis) {
		fprintf(stderr, "invalid xform command, must have both -rot-angle and -rot-axis or neither\n");
		return false;
	}

	Object *obj = 0;

	int nobj = scn->get_object_count();
	for(int i=0; i<nobj; i++) {
		Object *tmp = scn->get_object(i);
		if(strcmp(tmp->get_name(), name) == 0) {
			obj = tmp;
			break;
		}
	}

	if(!obj) {
		fprintf(stderr, "invalid xform, refers to nonexistent object: %s\n", name);
		return false;
	}

	if(have_pos) {
		obj->set_position(pos, tm);
	}
	if(have_rot_angle) {
		obj->set_rotation(Quaternion(rot_axis, rot_angle), tm);
	}
	if(have_scale) {
		obj->set_scaling(scale, tm);
	}

	if(have_extrap) {
		obj->set_extrapolator(extrap);
	}
	return true;
}

static bool parse_vec(char **argv, Vector3 *vptr)
{
	char *endp;

	vptr->x = strtod(argv[0], &endp);
	if(endp == argv[0])
		return false;
	vptr->y = strtod(argv[1], &endp);
	if(endp == argv[1])
		return false;
	vptr->z = strtod(argv[2], &endp);
	if(endp == argv[2])
		return false;

	return true;
}


static char *strip_space(char *s)
{
	while(*s && isspace(*s)) s++;

	if(!*s)
		return s;

	char *endp = s + strlen(s) - 1;
	while(isspace(*endp)) endp--;
	endp[1] = 0;

	if((endp = strrchr(s, '#'))) {
		*endp = 0;
	}

	return s;
}

static char **split_args(char *args)
{
	static std::vector<char*> argv;
	char *tok = 0;

	argv.clear();

	while((tok = strtok(tok ? 0 : args, " \t\v\n\r"))) {
		argv.push_back(tok);
	}
	argv.push_back(0);

	return &argv[0];
}
