/*
Simple introductory ray tracer
Copyright (C) 2012  John Tsiombikas <nuclear@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef SCENE_H_
#define SCENE_H_

#include <vector>
#include "object.h"
#include "light.h"
#include "camera.h"
#include "texture.h"

class Scene {
private:
	std::vector<Object*> objects;
	std::vector<Light*> lights;
	Camera *camera;

	static TargetCamera def_cam;	// default camera

	Color bgcolor;
	TextureCube *envmap, *envmap_conv;
	float fog_start, fog_end;

public:
	Scene();
	~Scene();

	bool load(const char *fname);
	bool save(const char *fname) const;

	void set_background_color(const Color &color);
	void set_fog(float fog_start, float fog_end);

	void set_environment_map(TextureCube *map, TextureCube *map_conv = 0);

	void add_object(Object *obj);
	Object *get_object(int i) const;
	int get_object_count() const;

	void add_light(Light *lt);
	Light *get_light(int i) const;
	int get_light_count() const;

	void set_camera(Camera *cam);
	Camera *get_camera() const;

	bool intersect(const Ray &ray, HitPoint *hit) const;

	Color env_color(const Ray &ray) const;

	void prepare_xform(long msec);

	friend Color trace_ray(const Scene *scn, const Ray &ray, int rdepth);
	friend Color shade(const Scene *scn, const Ray &ray, const HitPoint &hit, int rdepth);
};

#endif	// SCENE_H_
