/*
Simple introductory ray tracer
Copyright (C) 2012  John Tsiombikas <nuclear@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <algorithm>
#include "scene.h"
#include "texture.h"

bool load_scene_file(Scene *scn, const char *fname);
bool save_scene_file(const Scene *scn, const char *fname);


// default camera
TargetCamera Scene::def_cam(Vector3(0, 0, -10), Vector3(0, 0, 0));

Scene::Scene()
{
	camera = &def_cam;
	bgcolor = Color(0, 0, 0);
	envmap = envmap_conv = 0;
	fog_start = fog_end = -1;
}

Scene::~Scene()
{
	for(auto obj: objects) {
		delete obj;
	}
	for(auto lt: lights) {
		delete lt;
	}
	if(camera != &def_cam) {
		delete camera;
	}

	delete envmap;
	if(envmap_conv != envmap) {
		delete envmap_conv;
	}
}

bool Scene::load(const char *fname)
{
	return load_scene_file(this, fname);
}

bool Scene::save(const char *fname) const
{
	return save_scene_file(this, fname);
}

void Scene::set_background_color(const Color &color)
{
	bgcolor = color;
}

void Scene::set_fog(float fog_start, float fog_end)
{
	this->fog_start = fog_start;
	this->fog_end = fog_end;
}

void Scene::set_environment_map(TextureCube *map, TextureCube *map_conv)
{
	envmap = map;
	envmap_conv = map_conv;//map_conv ? map_conv : map;
}

void Scene::add_object(Object *obj)
{
	objects.push_back(obj);
}

Object *Scene::get_object(int i) const
{
	if(i < 0 || i >= (int)objects.size()) {
		return 0;
	}
	return objects[i];
}

int Scene::get_object_count() const
{
	return (int)objects.size();
}

void Scene::add_light(Light *lt)
{
	lights.push_back(lt);
}

Light *Scene::get_light(int i) const
{
	if(i < 0 || i >= (int)lights.size()) {
		return 0;
	}
	return lights[i];
}

int Scene::get_light_count() const
{
	return (int)lights.size();
}

void Scene::set_camera(Camera *cam)
{
	if(camera != &def_cam) {
		delete camera;
	}
	camera = cam;
}

Camera *Scene::get_camera() const
{
	return camera;
}

bool Scene::intersect(const Ray &ray, HitPoint *nearest_hit) const
{
	nearest_hit->obj = 0;
	nearest_hit->dist = FLT_MAX;

	// find the nearest hit (if any)
	for(Object *obj: objects) {
		HitPoint hit;
		if(obj->intersect(ray, &hit) && hit.dist < nearest_hit->dist) {
			*nearest_hit = hit;
		}
	}
	return nearest_hit->obj != 0;
}

Color Scene::env_color(const Ray &ray) const
{
	if(envmap) {
		Vector3 dir = ray.dir.normalized();
		return envmap->sample(dir.x, dir.y, dir.z);
	}
	return bgcolor;
}

void Scene::prepare_xform(long msec)
{
	int nobj = get_object_count();
	for(int i=0; i<nobj; i++) {
		objects[i]->prepare_xform(msec);
	}
}
