/*
Simple introductory ray tracer
Copyright (C) 2012  John Tsiombikas <nuclear@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "plane.h"

Plane::Plane()
	: normal(0.0, 1.0, 0.0)
{
	dist = 0.0;
}

Plane::Plane(const Vector3 &norm, float dist)
	: normal(norm)
{
	normal.normalize();
	this->dist = dist;
}

bool Plane::intersect(const Ray &inray, HitPoint *pt) const
{
	Ray ray = inray.transformed(inv_xform);

	float ndotdir = dot_product(normal, ray.dir);
	if(fabs(ndotdir) < 1e-5) {
		return false;	// ray parallel to the plane
	}

	Vector3 planept = normal * dist;
	Vector3 pptdir = planept - ray.origin;

	float t = dot_product(normal, pptdir) / ndotdir;
	if(t < 1e-4) {
		return false;	// discard intersections behind the origin
	}

	// fill the HitPoint structure
	pt->obj = this;
	pt->dist = t;

	pt->pos = ray.origin + ray.dir * t;

	pt->texcoord.x = pt->pos.x;
	pt->texcoord.y = pt->pos.z;

	pt->pos.transform(xform);
	pt->normal = normal.transformed(dir_xform);//.normalized();
	return true;
}
