#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>

#include <GL/glew.h>

#ifndef __APPLE__
#include <GL/glut.h>
#else
#include <GLUT/glut.h>
#endif

#include "scene.h"
#include "sphere.h"
#include "plane.h"
#include "image.h"
#include "rend.h"
#include "glsdr.h"

static bool init(const char *scene_fname);
static void cleanup();
static void disp();
static void draw_text(float r, float g, float b, const char *fmt, ...);
static void update_texture(const float *fb);
static void idle();
static void reshape(int x, int y);
static void handle_keys(float dt);
static void keyb(unsigned char key, int x, int y);
static void keyb_up(unsigned char key, int x, int y);
static void mouse(int bn, int st, int x, int y);
static void motion(int x, int y);
static void sball_motion(int x, int y, int z);
static void sball_rotate(int x, int y, int z);
static void sball_button(int bn, int state);
static int next_pow2(int x);

static unsigned int tex;
static long last_fps_upd, first_time = -1;
static long frames, total_frames;
static float fps;
static int xsz = 800;
static int ysz = 450;
static int tex_xsz, tex_ysz;

static Scene *scn;
static FlyCamera *cam;

static unsigned int post_sdr;

static float img_scale = 1.0f;
static bool keystate[256];

int main(int argc, char **argv)
{
	glutInit(&argc, argv);

	char *fname = 0;
	int num_samples = 1;

	for(int i=1; i<argc; i++) {
		if(argv[i][0] == '-') {
			if(strcmp(argv[i], "-size") == 0) {
				if(sscanf(argv[++i], "%dx%d", &xsz, &ysz) < 2) {
					fprintf(stderr, "-size must be followed by the image resolution (WxH)\n");
					return 1;
				}

			} else if(strcmp(argv[i], "-samples") == 0) {
				num_samples = atoi(argv[++i]);
				if(num_samples <= 0) {
					fprintf(stderr, "-samples must be followed by the number of samples per pixel\n");
					return 1;
				}
			} else if(strcmp(argv[i], "-scale") == 0) {
				img_scale = atof(argv[++i]);
				if(img_scale <= 1.0f) {
					fprintf(stderr, "-scale must be followed by an image scale factor >= 1\n");
					return 1;
				}
			} else {
				fprintf(stderr, "invalid option: %s\n", argv[i]);
				return 1;
			}
		} else {
			if(fname) {
				fprintf(stderr, "unexpected argument: %s\n", argv[i]);
				return 1;
			}
			fname = argv[i];
		}
	}

	glutInitWindowSize(xsz * img_scale, ysz * img_scale);
	glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE);
	glutCreateWindow("single threaded");

	glutDisplayFunc(disp);
	glutIdleFunc(idle);
	glutReshapeFunc(reshape);
	glutKeyboardFunc(keyb);
	glutKeyboardUpFunc(keyb_up);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutSpaceballMotionFunc(sball_motion);
	glutSpaceballRotateFunc(sball_rotate);
	glutSpaceballButtonFunc(sball_button);

	glewInit();

	if(!init(fname)) {
		return 1;
	}
	atexit(cleanup);

	set_render_samples(num_samples);

	glutMainLoop();
	return 0;
}

static bool init(const char *scene_fname)
{
	scn = new Scene;
	if(!scn->load(scene_fname ? scene_fname : "scene")) {
		return false;
	}

	cam = new FlyCamera;
	cam->input_move(0, 1.5, -10);
	cam->input_rotate(25, 0, 0);
	scn->set_camera(cam);

	if(!init_renderer(scn, xsz, ysz)) {
		return false;
	}

	glGenTextures(1, &tex);
	glBindTexture(GL_TEXTURE_2D, tex);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	int srgb_capable;
	if(GLEW_EXT_framebuffer_sRGB && (glGetIntegerv(GL_FRAMEBUFFER_SRGB_CAPABLE_EXT, &srgb_capable), srgb_capable)) {
		printf("enabling sRGB framebuffer\n");
		glEnable(GL_FRAMEBUFFER_SRGB);
	} else {
		printf("using post shader for gamma correction\n");
		post_sdr = create_program_load(0, "data/postsdr.glsl");
	}

	last_fps_upd = glutGet(GLUT_ELAPSED_TIME);
	first_time = last_fps_upd;
	return true;
}

static void cleanup()
{
	long interval = glutGet(GLUT_ELAPSED_TIME) - first_time;
	printf("average fps: %.2f\n", (float)total_frames / ((float)interval / 1000.0f));

	destroy_renderer();
}

static void disp()
{
	static long prev_msec;
	long interval, msec = glutGet(GLUT_ELAPSED_TIME);

	handle_keys((msec - prev_msec) / 1000.0f);
	prev_msec = msec;

	float *fb = render_frame(msec);

	update_texture(fb);

	glUseProgram(post_sdr);
	glEnable(GL_TEXTURE_2D);
	glBegin(GL_QUADS);
	glTexCoord2f(0, 1);
	glVertex2f(-1, -1);
	glTexCoord2f(1, 1);
	glVertex2f(1, -1);
	glTexCoord2f(1, 0);
	glVertex2f(1, 1);
	glTexCoord2f(0, 0);
	glVertex2f(-1, 1);
	glEnd();
	glDisable(GL_TEXTURE_2D);

	draw_text(0.8, 0.75, 0, "fps: %.2f", fps);

	glutSwapBuffers();
	frames++;
	total_frames++;

	interval = (msec = glutGet(GLUT_ELAPSED_TIME)) - last_fps_upd;
	if(interval >= 2000) {
		float tm = (float)interval / 1000.0f;
		fps = (float)frames / tm;
		/*printf("%.2f fps         \r", fps);
		fflush(stdout);*/
		last_fps_upd = msec;
		frames = 0;
	}
}

void draw_text(float r, float g, float b, const char *fmt, ...)
{
	char buf[256], *text = buf;
	va_list ap;

	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	glUseProgram(0);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, xsz, 0, ysz, -1, 1);

	glColor3f(r, g, b);
	glRasterPos2f(2, 4);
	while(*text) {
		glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, *text++);
	}
	glColor3f(1, 1, 1);

	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}


static void update_texture(const float *fb)
{
	int ntx = next_pow2(xsz);
	int nty = next_pow2(ysz);

	if(ntx != tex_xsz || nty != tex_ysz) {
		tex_xsz = ntx;
		tex_ysz = nty;

		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F, tex_xsz, tex_ysz, 0, GL_RGB, GL_FLOAT, 0);
	}

	glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, xsz, ysz, GL_RGB, GL_FLOAT, fb);
}

static void idle()
{
	glutPostRedisplay();
}

static void reshape(int x, int y)
{
	int tx, ty;

	xsz = x / img_scale;
	ysz = y / img_scale;

	glViewport(0, 0, x, y);

	/* setup the texture matrix that maps just the visible area
	 * of the texture to [0, 1]
	 */
	tx = next_pow2(xsz);
	ty = next_pow2(ysz);

	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();
	glScalef((float)xsz / tx, (float)ysz / ty, 1.0f);

	resize_renderer(xsz, ysz);
}

static void handle_keys(float dt)
{
	Vector3 move;
	float tilt = 0.0f;
	float offs = dt * 8.0;

	if(keystate['w']) {
		move.z += offs;
	}
	if(keystate['s']) {
		move.z -= offs;
	}
	if(keystate['d']) {
		move.x += offs;
	}
	if(keystate['a']) {
		move.x -= offs;
	}

	if(keystate['q']) {
		tilt -= dt;
	}
	if(keystate['e']) {
		tilt += dt;
	}

	cam->input_move(move.x, move.y, move.z);
	cam->input_rotate(0, 0, tilt);
}

static void keyb(unsigned char key, int x, int y)
{
	keystate[key] = true;

	switch(key) {
	case 27:
		exit(0);
	}
}

static void keyb_up(unsigned char key, int x, int y)
{
	keystate[key] = false;
}

static int prev_x, prev_y;

static void mouse(int bn, int st, int x, int y)
{
	prev_x = x;
	prev_y = y;
}

static void motion(int x, int y)
{
	int dx = x - prev_x;
	int dy = y - prev_y;
	prev_x = x;
	prev_y = y;
	cam->input_rotate(-dy * 0.01, -dx * 0.01,  0);
}

static void sball_motion(int x, int y, int z)
{
	float fx = x * 0.01;
	float fy = y * 0.01;
	float fz = z * 0.01;
	cam->input_move(fx, fy, fz);
}

static void sball_rotate(int x, int y, int z)
{
	float fx = x * 0.00025;
	float fy = y * 0.00025;
	float fz = z * 0.00025;
	cam->input_rotate(fx, fy, fz);
}

static void sball_button(int bn, int state)
{
}


static int next_pow2(int x)
{
	x--;
	x = (x >> 1) | x;
	x = (x >> 2) | x;
	x = (x >> 4) | x;
	x = (x >> 8) | x;
	x = (x >> 16) | x;
	return x + 1;
}
